// Listen for messages from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.type === 'FETCH_SERP') {
        fetchSERP(request.keyword, request.domain, request.lang, request.geo)
            .then(data => sendResponse({ success: true, rank: data.rank, results: data.results }))
            .catch(error => sendResponse({ success: false, error: error.message }));
        return true; // Will respond asynchronously
    }

    if (request.type === 'CHECK_BACKLINK') {
        // Now accepts userAgent
        checkBacklink(request.url, request.targetDomain, request.userAgent)
            .then(data => sendResponse({ success: true, ...data }))
            .catch(error => sendResponse({ success: false, error: error.message }));
        return true;
    }

    if (request.type === 'FETCH_URI_CONTENT') {
        fetchUriContent(request.url, request.userAgent)
            .then(content => sendResponse({ success: true, content }))
            .catch(error => sendResponse({ success: false, error: error.message }));
        return true;
    }
});

// Manage Offscreen Document
let creating; // A global promise to avoid concurrency issues
async function setupOffscreenDocument(path) {
    // Check if offscreen document exists
    const existingContexts = await chrome.runtime.getContexts({
        contextTypes: ['OFFSCREEN_DOCUMENT'],
        documentUrls: [path]
    });

    if (existingContexts.length > 0) {
        return;
    }

    // create offscreen document
    if (creating) {
        await creating;
    } else {
        creating = chrome.offscreen.createDocument({
            url: path,
            reasons: ['DOM_PARSER'],
            justification: 'Parse SERP HTML to extract ranks'
        });

        try {
            await creating;
        } catch (e) {
            // Ignore error if document already exists (race condition)
            if (!e.message.includes('Only a single offscreen document may be created')) {
                throw e;
            }
        }

        creating = null;
    }
}

async function fetchSERP(keyword, domain, lang = 'en', geo = 'us') {
    const pagesToFetch = [0, 10, 20]; // Fetch first 3 pages
    let allResults = [];

    for (const start of pagesToFetch) {
        const url = `https://www.google.com/search?q=${encodeURIComponent(keyword)}&gl=${geo}&hl=${lang}&start=${start}`;

        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 10000); // 10s per page

        try {
            const response = await fetch(url, {
                signal: controller.signal,
                headers: {
                    'User-Agent': 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
                }
            });
            clearTimeout(timeoutId);

            if (!response.ok) throw new Error(`Failed to fetch Google Search (start=${start})`);

            const html = await response.text();

            // Use offscreen document to parse
            await setupOffscreenDocument('offscreen.html');

            // Send HTML to offscreen document
            const parsedData = await chrome.runtime.sendMessage({
                type: 'PARSE_SERP',
                html: html,
                domain: domain
            });

            if (parsedData && parsedData.results) {
                allResults = allResults.concat(parsedData.results);
            }

            // Random delay between pages (1-2s)
            await new Promise(r => setTimeout(r, 1000 + Math.random() * 1000));

        } catch (e) {
            console.error(`Error fetching page start=${start}`, e);
            // Continue to next page even if one fails
        }
    }

    // Calculate Rank from combined results
    let rank = 0;
    for (let i = 0; i < allResults.length; i++) {
        if (allResults[i].includes(domain)) {
            rank = i + 1;
            break;
        }
    }

    return { rank, results: allResults };
}

async function checkBacklink(url, targetDomain, userAgent) {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 15000);

    // Default UA if not provided
    const uaToUse = userAgent || 'Mozilla/5.0 (compatible; FreeSEOToolsBot/1.0)';

    try {
        const response = await fetch(url, {
            signal: controller.signal,
            headers: {
                'User-Agent': uaToUse
            }
        });

        if (!response.ok) throw new Error(`Failed to fetch URL: ${response.status}`);
        const html = await response.text();

        await setupOffscreenDocument('offscreen.html');

        const result = await chrome.runtime.sendMessage({
            type: 'PARSE_BACKLINK',
            html: html,
            targetDomain: targetDomain
        });

        return result; // { found, anchor, rel }

    } catch (e) {
        console.error('Backlink check error', e);
        throw e;
    }
}

async function fetchUriContent(url, userAgent) {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 10000);
    const uaToUse = userAgent || 'Mozilla/5.0 (compatible; FreeSEOToolsBot/1.0)';

    try {
        const response = await fetch(url, {
            signal: controller.signal,
            headers: {
                'User-Agent': uaToUse
            }
        });
        clearTimeout(timeoutId);

        if (!response.ok) throw new Error(`HTTP Error: ${response.status}`);

        const text = await response.text();

        // Extract headers
        const headers = {};
        for (const [key, value] of response.headers.entries()) {
            headers[key] = value;
        }

        return { content: text, headers: headers };
    } catch (e) {
        throw e;
    }
}
