console.log('SERP Extension: Content script loaded');

// Notify web app that extension is ready via CustomEvent on document (shared DOM)
// We use a timeout to ensure the web app's listeners are attached if it loads simultaneously
setTimeout(() => {
    document.dispatchEvent(new CustomEvent('SERP_EXTENSION_READY'));
    console.log('SERP Extension: Ready event dispatched');
}, 100);

// Also listen for a ping from the app (in case app loaded first)
document.addEventListener('SERP_EXTENSION_PING', () => {
    console.log('SERP Extension: Ping received, sending Ready');
    document.dispatchEvent(new CustomEvent('SERP_EXTENSION_READY'));
});

// Listen for requests from the web app
document.addEventListener('SERP_CHECK_REQUEST', (event) => {
    console.log('SERP Extension: Received request', event.detail);
    const { keyword, domain, lang, geo } = event.detail;

    // Send to background script
    // Send to background script
    try {
        if (!chrome.runtime?.id) {
            throw new Error('Extension context invalidated');
        }

        chrome.runtime.sendMessage({ type: 'FETCH_SERP', keyword, domain, lang, geo }, (response) => {
            // Check for runtime errors (like context invalidated during request)
            if (chrome.runtime.lastError) {
                console.error('SERP Extension: Runtime error', chrome.runtime.lastError);
                document.dispatchEvent(new CustomEvent('SERP_CHECK_RESPONSE', {
                    detail: { success: false, error: 'EXTENSION_INVALIDATED' }
                }));
                return;
            }

            console.log('SERP Extension: Received background response', response);
            // Send response back to web app
            document.dispatchEvent(new CustomEvent('SERP_CHECK_RESPONSE', {
                detail: response
            }));
        });
    } catch (e) {
        console.error('SERP Extension: SendMessage error', e);
        document.dispatchEvent(new CustomEvent('SERP_CHECK_RESPONSE', {
            detail: { success: false, error: 'EXTENSION_INVALIDATED' }
        }));
    }
});

// Listen for backlink check requests
document.addEventListener('BACKLINK_CHECK_REQUEST', (event) => {
    console.log('Backlink Check: Received request', event.detail);
    const { url, targetDomain, userAgent } = event.detail;

    try {
        if (!chrome.runtime?.id) throw new Error('Extension context invalidated');

        chrome.runtime.sendMessage({ type: 'CHECK_BACKLINK', url, targetDomain, userAgent }, (response) => {
            if (chrome.runtime.lastError) {
                console.error('Backlink Check: Runtime error', chrome.runtime.lastError);
                document.dispatchEvent(new CustomEvent('BACKLINK_CHECK_RESPONSE', {
                    detail: { success: false, error: 'EXTENSION_INVALIDATED' }
                }));
                return;
            }

            console.log('Backlink Check: Received background response', response);
            document.dispatchEvent(new CustomEvent('BACKLINK_CHECK_RESPONSE', {
                detail: response
            }));
        });
    } catch (e) {
        console.error('Backlink Check: SendMessage error', e);
        document.dispatchEvent(new CustomEvent('BACKLINK_CHECK_RESPONSE', {
            detail: { success: false, error: 'EXTENSION_INVALIDATED' }
        }));
    }
});

// Listen for generic content fetch requests
document.addEventListener('FETCH_CONTENT_REQUEST', (event) => {
    console.log('Fetch Content: Received request', event.detail);
    const { url, userAgent } = event.detail;

    try {
        if (!chrome.runtime?.id) throw new Error('Extension context invalidated');

        chrome.runtime.sendMessage({ type: 'FETCH_URI_CONTENT', url, userAgent }, (response) => {
            if (chrome.runtime.lastError) {
                console.error('Fetch Content: Runtime error', chrome.runtime.lastError);
                document.dispatchEvent(new CustomEvent('FETCH_CONTENT_RESPONSE', {
                    detail: { success: false, error: 'EXTENSION_INVALIDATED' }
                }));
                return;
            }

            document.dispatchEvent(new CustomEvent('FETCH_CONTENT_RESPONSE', {
                detail: response || { success: false, error: 'No response from background' }
            }));
        });
    } catch (e) {
        document.dispatchEvent(new CustomEvent('FETCH_CONTENT_RESPONSE', {
            detail: { success: false, error: 'EXTENSION_INVALIDATED' }
        }));
    }
});

// Universal Proxy Listener for Future Tools
document.addEventListener('FREESEOTOOLS_PROXY_REQUEST', (event) => {
    console.log('Proxy Request:', event.detail);
    const { action, params, requestId } = event.detail;

    if (!action) {
        console.error('Proxy Request: Missing action');
        return;
    }

    try {
        if (!chrome.runtime?.id) throw new Error('Extension context invalidated');

        // Forward to background script
        // We construct the message by spreading params and adding the type
        const message = { type: action, ...params };

        chrome.runtime.sendMessage(message, (response) => {
            if (chrome.runtime.lastError) {
                console.error('Proxy Request: Runtime error', chrome.runtime.lastError);
                document.dispatchEvent(new CustomEvent('FREESEOTOOLS_PROXY_RESPONSE', {
                    detail: { success: false, error: 'EXTENSION_INVALIDATED', requestId }
                }));
                return;
            }

            document.dispatchEvent(new CustomEvent('FREESEOTOOLS_PROXY_RESPONSE', {
                detail: { ...response, requestId }
            }));
        });
    } catch (e) {
        document.dispatchEvent(new CustomEvent('FREESEOTOOLS_PROXY_RESPONSE', {
            detail: { success: false, error: 'EXTENSION_INVALIDATED', requestId }
        }));
    }
});
