chrome.runtime.onMessage.addListener(handleMessages);

function handleMessages(message, sender, sendResponse) {
    if (message.type === 'PARSE_SERP') {
        const results = parseSerp(message.html, message.domain);
        sendResponse(results);
    }
    if (message.type === 'PARSE_BACKLINK') {
        const results = parseBacklink(message.html, message.targetDomain);
        sendResponse(results);
    }
}

function parseBacklink(htmlString, targetDomain) {
    const parser = new DOMParser();
    const doc = parser.parseFromString(htmlString, 'text/html');

    // Find all links
    const links = Array.from(doc.querySelectorAll('a[href]'));
    const matches = [];

    for (const link of links) {
        try {
            const href = link.getAttribute('href');
            // Check if absolute or relative (resolve relative if needed, but usually we look for domain)
            // If href contains targetDomain
            if (href && (href.includes(targetDomain) || (new URL(href, 'http://example.com').hostname.includes(targetDomain)))) {
                matches.push({
                    anchor: link.textContent.trim().substring(0, 100) || '[Image/No Text]',
                    rel: link.getAttribute('rel') || 'dofollow',
                    href: href
                });
            }
        } catch (e) {
            // Ignore URL parse errors
        }
    }

    if (matches.length > 0) {
        return { found: true, matches: matches };
    }

    return { found: false };
}

function parseSerp(htmlString, targetDomain) {
    const parser = new DOMParser();
    const doc = parser.parseFromString(htmlString, 'text/html');

    // Google's main results are usually in div.g or div.MjjYud
    // We look for containers that have an anchor tag with an href
    // that is NOT a Google internal link.

    // Strategy:
    // 1. Find all 'a' tags.
    // 2. Filter out Google internal links.
    // 3. Group by visual container? 
    // Actually, the user wants "Rank by Site Position".
    // This means we should find the top-level result containers.

    // A robust way is to look for the main column results.
    // Usually inside #search or #rso
    const rso = doc.getElementById('rso');
    if (!rso) {
        // Fallback if ID changes, try finding the main container by class or structure
        // But for now, let's try a broad search for result containers if #rso is missing
        return parseByContainers(doc.body, targetDomain);
    }

    return parseByContainers(rso, targetDomain);
}

function parseByContainers(rootElement, targetDomain) {
    let rank = 0;
    let organicCount = 0;
    let results = [];

    // Heuristic: Google results ALWAYS have an H3 title.
    // We find all H3s, then find the main link associated with them.
    const titles = Array.from(rootElement.querySelectorAll('h3'));

    const processedLinks = new Set();

    for (const title of titles) {
        // The link is usually the parent 'a' or a sibling 'a'.
        // In modern Google, the H3 is inside the 'a' or the 'a' wraps the container.
        let link = title.closest('a');

        // If H3 is not inside an 'a', look for the 'a' that contains the URL just above/below
        if (!link) {
            // Sometimes structure is: div > div > a > h3 (handled by closest)
            // Or: div > a ... div > h3
            // Let's look for the closest container .g and find the first a[href]
            const container = title.closest('.g') || title.closest('div[jscontroller]') || title.parentElement.parentElement;
            if (container) {
                link = container.querySelector('a[href]');
            }
        }

        if (!link) continue;

        const href = link.getAttribute('href');
        if (!href) continue;

        // Dedup: If we already counted this URL (or very similar), skip.
        // Google sometimes puts multiple links to same place.
        if (processedLinks.has(href)) continue;

        // Filter Google links
        if (isGoogleLink(href)) continue;

        processedLinks.add(href);
        organicCount++;
        results.push(href);

        if (rank === 0 && href.includes(targetDomain)) {
            rank = organicCount;
        }

        if (organicCount >= 100) break;
    }

    return { rank, results };
}

function isGoogleLink(url) {
    try {
        const urlObj = new URL(url, 'https://www.google.com'); // Handle relative URLs
        const hostname = urlObj.hostname;

        return /^(www\.)?google\.[a-z.]{2,10}$/i.test(hostname) ||
            hostname.includes('googleusercontent.com') ||
            hostname.includes('youtube.com') ||
            url.startsWith('/search') ||
            url.startsWith('/url');
    } catch (e) {
        return true; // If invalid URL, ignore it
    }
}
